unit CueDOMGen1;

{
  Demonstrate the generation of an XML document from a database
  using the CUESoft Document Object Model (DOM).
  Requires 'movie-watcher' alias to be set up in BDE.
  Requires CUEXml v2 package from CUESoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written October 10, 2000.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DBTables, StdCtrls, ExtCtrls, XmlObjModel, CommonXML;

type
  TfrmDOMXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
  public
  end;

var
  frmDOMXML: TfrmDOMXML;

implementation

{$R *.DFM}

{ Generate the XML document as text }
procedure TfrmDOMXML.btnGenerateClick(Sender: TObject);
var
  XMLModel: TXmlObjModel;
  XMLDoc: TXmlDocument;
  XMLText: string;
  Posn: Integer;

  { Add a simple element that only contains text }
  procedure AddSimpleElement(Parent: TXmlElement; Field: TField;
    AsCDATA: Boolean = False);
  var
    Internal: TXmlElement;
  begin
    Internal := XMLDoc.CreateElement(ModifyName(Field.FieldName));
    Parent.AppendChild(Internal);
    if AsCDATA then
      Internal.AppendChild(XMLDoc.CreateCDATASection(Field.DisplayText))
    else
      Internal.AppendChild(XMLDoc.CreateTextNode(Field.DisplayText));
  end;

  { Include attributes only if present }
  procedure AddOptAttribute(Element: TXmlElement; Field: TField);
  begin
    if Field.AsString <> '' then
      Element.SetAttribute(ModifyName(Field.FieldName), Field.DisplayText);
  end;

  { Include empty field tag only if flag in DB set }
  procedure AddOptElement(Parent: TXmlElement; Field: TField);
  begin
    if Field.AsBoolean then
      Parent.AppendChild(
        Parent.OwnerDocument.CreateElement(ModifyName(Field.FieldName)));
  end;

  { Generate XML prolog, style sheet reference, and main element }
  procedure GenerateHeaders;
  begin
    with XMLDoc do
    begin
      AppendChild(CreateProcessingInstruction(XMLTag, XMLPrologAttrs));
      AppendChild(CreateComment(XMLComment));
      AppendChild(CreateProcessingInstruction(XMLStyleTag, XMLStyleAttrs));
      AppendChild(CreateElement(MovieWatcherTag));
    end;
  end;

  { Compile elements for the stars of the movie }
  procedure GenerateStars(Starring: TXmlElement);
  begin
    with datCommonXML.qryStars do
    begin
      First;
      while not EOF do
      begin
        AddSimpleElement(Starring, FieldByName(StarField));
        Next;
      end;
    end;
  end;

  { Generate elements for each movie }
  procedure GenerateMovies;
  var
    Movies, Movie, Starring: TXmlElement;
  begin
    Movies := XMLDoc.CreateElement(MoviesTag);
    XMLDoc.DocumentElement.AppendChild(Movies);
    with datCommonXML.qryMovie do
    begin
      First;
      while not EOF do
      begin
        Movie := XMLDoc.CreateElement(MovieTag);
        Movies.AppendChild(Movie);
        Movie.SetAttribute(Id, FieldByName(MovieIdField).DisplayText);
        Movie.SetAttribute(Rating, FieldByName(RatingField).DisplayText);
        AddOptAttribute(Movie, FieldByName(LogoURLField));
        AddOptAttribute(Movie, FieldByName(URLField));
        AddSimpleElement(Movie, FieldByName(NameField));
        AddSimpleElement(Movie, FieldByName(LengthField));
        AddSimpleElement(Movie, FieldByName(DirectorField));
        Starring := XMLDoc.CreateElement(StarringTag);
        Movie.AppendChild(Starring);
        GenerateStars(Starring);
        AddSimpleElement(Movie, FieldByName(SynopsisField), True);
        Next;
      end;
    end;
  end;

  { Compile elements for the pricing schemes }
  procedure GeneratePricing(Pricing: TXmlElement);
  var
    Price: TXmlElement;
  begin
    with datCommonXML.qryPricing do
    begin
      First;
      while not EOF do
      begin
        Price := XMLDoc.CreateElement(PriceTag);
        Pricing.AppendChild(Price);
        Price.SetAttribute(Id, FieldByName(PricingIdField).DisplayText);
        AddSimpleElement(Price, FieldByName(NameField));
        AddSimpleElement(Price, FieldByName(PeriodField));
        AddSimpleElement(Price, FieldByName(AdultField));
        AddSimpleElement(Price, FieldByName(ChildField));
        if FieldByName(DiscountField).AsFloat <> 0 then
          AddSimpleElement(Price, FieldByName(DiscountField));
        Next;
      end;
    end;
  end;

  { Generate elements for each cinema }
  procedure GenerateCinemas;
  var
    Cinemas, Cinema, Facilities, Pricing: TXmlElement;
  begin
    Cinemas := XMLDoc.CreateElement(CinemasTag);
    XMLDoc.DocumentElement.AppendChild(Cinemas);
    with datCommonXML.qryCinema do
    begin
      First;
      while not EOF do
      begin
        Cinema := XMLDoc.CreateElement(CinemaTag);
        Cinemas.AppendChild(Cinema);
        Cinema.SetAttribute(Id, FieldByName(CinemaIdField).DisplayText);
        AddSimpleElement(Cinema, FieldByName(NameField));
        AddSimpleElement(Cinema, FieldByName(PhoneField));
        AddSimpleElement(Cinema, FieldByName(AddressField));
        AddSimpleElement(Cinema, FieldByName(DirectionsField));
        Facilities := XMLDoc.CreateElement(FacilitiesTag);
        Cinema.AppendChild(Facilities);
        AddOptElement(Facilities, FieldByName(CandyBarField));
        AddOptElement(Facilities, FieldByName(DisabledField));
        Pricing := XMLDoc.CreateElement(PricingTag);
        Cinema.AppendChild(Pricing);
        GeneratePricing(Pricing);
        Next;
      end;
    end;
  end;

  { Compile elements for the sessions for each screening }
  procedure GenerateSessions(Sessions: TXmlElement);
  var
    Session: TXmlElement;
  begin
    with datCommonXML.qrySessions do
    begin
      First;
      while not EOF do
      begin
        Session := XMLDoc.CreateElement(SessionTag);
        Sessions.AppendChild(Session);
        Session.SetAttribute(PricingId,
          FieldByName(PricingIdField).DisplayText);
        AddSimpleElement(Session, FieldByName(TimeField));
        Next;
      end;
    end;
  end;

  { Generate elements for each screening }
  procedure GenerateScreenings;
  var
    Screenings, Screening, Internal, Sessions: TXmlElement;
  begin
    Screenings := XMLDoc.CreateElement(ScreeningsTag);
    XMLDoc.DocumentElement.AppendChild(Screenings);
    with datCommonXML.qryScreening do
    begin
      First;
      while not EOF do
      begin
        Screening := XMLDoc.CreateElement(ScreeningTag);
        Screenings.AppendChild(Screening);
        Screening.SetAttribute(MovieId, FieldByName(MovieIdField).DisplayText);
        Screening.SetAttribute(CinemaId,
          FieldByName(CinemaIdField).DisplayText);
        AddSimpleElement(Screening, FieldByName(StartDateField));
        AddSimpleElement(Screening, FieldByName(EndDateField));
        Internal := XMLDoc.CreateElement(FeaturesTag);
        Screenings.AppendChild(Internal);
        AddSimpleElement(Internal, FieldByName(DigSoundField));
        Internal := XMLDoc.CreateElement(RestrictionsTag);
        Screenings.AppendChild(Internal);
        AddOptElement(Internal, FieldByName(NoPassesField));
        Sessions := XMLDoc.CreateElement(SessionsTag);
        Screenings.AppendChild(Sessions);
        GenerateSessions(Sessions);
        Next;
      end;
    end;
  end;

begin
  Screen.Cursor       := crHourglass;
  btnGenerate.Enabled := False;
  try
    { Instantiate the DOM }
    XMLModel := TXmlObjModel.Create(nil);
    try
      XMLDoc   := XMLModel.Document;
      { Generate the structure }
      GenerateHeaders;
      GenerateMovies;
      GenerateCinemas;
      GenerateScreenings;
      { And convert to XML }
      XMLModel.FormattedOutput := True;
      XMLText                  := XMLModel.XMLDocument;
      { Remove any null characters CUEXml may have inserted }
      repeat
        Posn := Pos(#0, XMLText);
        if Posn > 0 then
          Delete(XMLText, Posn, 1);
      until Posn = 0;
      memXML.Lines.Text := XMLText;
    finally
      { Release the DOM }
      XMLModel.Free;
    end;
  finally
    btnGenerate.Enabled := True;
    Screen.Cursor       := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmDOMXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

end.
